# Claude Code Cheat Sheet

Quick reference for expert Claude Code usage. Keep this handy.

---

## Essential Files

```
~/.claude/
  CLAUDE.md                          # Global instructions (all projects)
  settings.json                      # MCP servers, hooks, permissions
  keybindings.json                   # Keyboard shortcuts
  projects/<hash>/memory/MEMORY.md   # Project-specific persistent memory

<project-root>/
  CLAUDE.md                          # Project instructions
  .claude/commands/                  # Custom skills (slash commands)
```

---

## Built-in Commands

```
/help          Show help
/plan          Enter plan mode (review before Claude acts)
/memory        View/edit project memory
/settings      Open settings
/clear         Clear conversation history
```

---

## Key Concepts (5-Second Version)

| Concept | What It Does | When to Use |
|---------|-------------|-------------|
| **CLAUDE.md** | Project rules Claude reads every session | Set it up once at project start |
| **Plan Mode** | Claude designs approach, you approve | Any non-trivial task (3+ files) |
| **Memory** | Knowledge persists across sessions | After discovering patterns |
| **Hooks** | Auto-run scripts on events | Notifications, validation, logging |
| **MCP Servers** | Give Claude new tools (DB, GitHub, etc.) | Need external API/data access |
| **Skills** | Reusable prompts via `/command` | Repetitive workflows |

---

## Prompting Patterns

### Bug Fixing
```
Context: [what it should do]
Symptom: [what's actually happening]
Location: file.ts:123
Fix: [scope constraint]
```

### Feature Development
```
Add [feature] to [file].
Follow the pattern from [existing similar feature].
Don't touch [files to avoid].
Test with [test cases].
```

### Code Review
```
Review the changes in this branch.
Check for: correctness, security, performance, style.
Output: Summary, Issues (critical/major/minor), Verdict.
```

### Refactoring
```
Refactor [target] to [desired state].
Preserve: [what must not change]
Scope: only [these files]
```

---

## CLAUDE.md Template (Minimal)

```markdown
# Project Name

## Tech Stack
- Language:
- Framework:
- Database:

## Project Structure
- /src/api — [description]
- /src/services — [description]

## Code Style
- [rule]
- [rule]

## Never Do This
- [forbidden pattern]
- [forbidden pattern]

## Development Commands
- Tests: `[command]`
- Dev: `[command]`
```

[Full template: templates/CLAUDE.md](templates/CLAUDE.md)

---

## Common CLAUDE.md Rules

**Communication:**
```markdown
## Communication Style
- Be concise. Prefer bullets over paragraphs.
- No emoji.
```

**Tools:**
```markdown
## Development Tools
- Package manager: bun (never npm)
- Linter: biome (never eslint)
```

**Git:**
```markdown
## Git Workflow
- Conventional commits: feat:, fix:, chore:
- Never push to main
- Never use --no-verify
```

**Architecture:**
```markdown
## Architecture Rules
- No business logic in routes — use services
- All DB access via repositories
- Services never call each other — use events
```

---

## MCP Server Quick Config

```json
{
  "mcpServers": {
    "github": {
      "command": "npx",
      "args": ["-y", "@modelcontextprotocol/server-github"],
      "env": { "GITHUB_PERSONAL_ACCESS_TOKEN": "ghp_..." }
    },
    "postgres": {
      "command": "npx",
      "args": ["-y", "@modelcontextprotocol/server-postgres", "postgresql://..."]
    }
  }
}
```

---

## Hook Quick Config

```json
{
  "hooks": {
    "Stop": [
      {
        "matcher": "",
        "hooks": [
          {
            "type": "command",
            "command": "powershell.exe -File ~/.claude/hooks/notify.ps1"
          }
        ]
      }
    ]
  }
}
```

**Exit codes:**
- `0` = success, continue
- `1` = warning, continue
- `2` = block Claude from proceeding

---

## Efficiency Patterns

**Parallel requests:**
```
Read these 3 files in parallel: A, B, C
Then tell me [question].
```

**Batching:**
```
Do all of these in one go:
1. [task]
2. [task]
3. [task]
```

**Research then implement:**
```
Phase 1: Research [topics]. Don't write code yet.
Phase 2: Based on research, implement [feature].
```

---

## Plan Mode Workflow

```
You:    "I need to [task]"
Claude: [presents plan]
You:    "Approved" OR "Change step 3 to [revision]"
Claude: [implements]
```

Always use for:
- 3+ file changes
- Architecture decisions
- Database migrations
- Git operations (merge, rebase)
- Deletions

---

## Memory Management

**Add to memory:**
```
"Remember: we use Prisma, not raw SQL"
```

**Check memory:**
```
"What do you know about this project from memory?"
```

**Correct memory:**
```
"Forget what you saved about X — we changed it to Y"
```

**Memory vs CLAUDE.md:**
- CLAUDE.md = stable rules you define upfront
- MEMORY.md = discoveries Claude makes during work

---

## Custom Skills (Slash Commands)

Create: `~/.claude/commands/skill-name.md`

```markdown
[Instructions for Claude when /skill-name is invoked]

Use $ARGUMENTS for user input after the command.
```

Invoke: `/skill-name some arguments`

**Popular skills:**
- `/commit` - Create conventional commit
- `/review` - Code review current branch
- `/debug` - Systematic debugging workflow
- `/explain` - Deep code explanation

---

## Troubleshooting Quick Fixes

| Problem | Fix |
|---------|-----|
| Claude asks known questions | Add answers to CLAUDE.md |
| Wrong architectural choice | Add architecture rules to CLAUDE.md |
| Went down wrong path | Use Plan Mode next time |
| Can't find files | Tell exact path, add to "Key Files" in CLAUDE.md |
| Tests broke | "Read test output and fix failures" |
| Too many changes | Constrain scope: "Only change X" |
| Wrong tool used | Add tool preferences to CLAUDE.md |
| Forgot earlier context | Start new session, save to MEMORY first |
| Too verbose | Add communication style to CLAUDE.md |

---

## Advanced Patterns

**Test-Driven Development:**
```
Here are the tests. Make them pass. Don't modify tests.
```

**Self-Review:**
```
Review your own implementation for:
correctness, edge cases, security, performance.
Suggest top 3 improvements.
```

**Reflection:**
```
Summarize what we built today.
What's incomplete? What's fragile?
What should be added to CLAUDE.md?
```

**Living Spec:**
Maintain `SPEC.md` with current requirements. Claude reads it.

**Architectural Decisions:**
Maintain `DECISIONS.md` with ADRs (Architecture Decision Records).

---

## Expert Habits

1. **Invest in CLAUDE.md** - Highest leverage activity
2. **Use Plan Mode** - For anything non-trivial
3. **Be specific** - File:line references, exact constraints
4. **Parallelize** - Ask for multiple things at once
5. **Maintain memory** - Capture learnings
6. **Update after mistakes** - Add rule to CLAUDE.md
7. **Review before trusting** - Self-review pattern
8. **Think in batches** - Combine independent tasks

---

## File Sizes to Watch

- **CLAUDE.md**: Keep under ~200 lines (loaded every session)
- **MEMORY.md**: Keep under ~200 lines (first 200 auto-loaded)
- Long memory? Break into topic files

---

## Getting Help

- In CLI: `/help`
- GitHub Issues: https://github.com/anthropics/claude-code/issues
- Documentation: Ask Claude "How do I [feature]?"

---

## The Core Loop

```
1. Notice issue
2. Add rule to CLAUDE.md
3. Issue never happens again
4. Repeat
```

Your productivity compounds over time.

---

**Print this. Keep it visible. Reference it daily.**
