# Lesson 03: Plan Mode & Approval Workflows

## What Is Plan Mode?

Plan Mode is a special state where Claude explores your codebase and designs an implementation strategy — **without writing any code or making any changes**. You review the plan and approve it before Claude does anything.

This is one of the most important habits you can build as a Claude Code user.

---

## Why Plan Mode Matters

Claude is eager and capable. Without plan mode, it will immediately start modifying files. For non-trivial tasks, this leads to:

- **Wasted work** — Claude implements the wrong approach and has to undo it
- **Unexpected side effects** — Changes to files you didn't realize were affected
- **Missed requirements** — Claude misunderstood what you wanted
- **Hard-to-reverse actions** — Deleted files, force pushes, schema migrations

Plan Mode is the "measure twice, cut once" principle applied to AI-assisted coding.

---

## How to Enter Plan Mode

**From the command line:**
```
/plan
```

**Ask Claude directly:**
> "Before making any changes, lay out your implementation plan."

**In your CLAUDE.md:**
```markdown
## Workflow
Always enter plan mode for any task that touches more than 2 files.
```

---

## What Happens in Plan Mode

1. Claude reads relevant files (Glob, Grep, Read)
2. Claude designs an approach
3. Claude presents a structured plan with:
   - What files will be changed and why
   - What new files will be created
   - The implementation steps in order
   - Any trade-offs or decisions you should be aware of
4. **You approve, modify, or reject the plan**
5. Claude implements exactly what was agreed

---

## Reading a Plan: What to Look For

When Claude presents a plan, check:

### Does it understand the task correctly?
> If Claude's plan description doesn't match your intent, correct it before it writes a single line.

### Are the file changes reasonable?
> Does it need to touch `auth.ts` to add a feature to `billing.ts`? Ask why.

### Is the scope right?
> Plans often include "while I'm in here, I'll also..." — push back if it's doing more than you asked.

### Are there architectural red flags?
> If Claude's plan adds a database call to a component, or puts business logic in a route handler, correct it now.

### What's missing?
> Does the plan include tests? Error handling? Migration scripts? Make sure it's complete.

---

## Modifying Plans Before Approval

You can (and should) push back on plans:

> "The plan looks good except step 3 — don't create a new service class, put that logic in the existing `UserService`."

> "You're touching too many files. Scope this down to just the API layer. We'll handle the frontend separately."

> "Before implementing, I want you to also add a plan for error handling. What happens if the external API call fails?"

---

## When to Use Plan Mode

**Always use plan mode when:**
- The task will touch more than 2-3 files
- You're making architectural decisions
- You're running database migrations
- You're modifying CI/CD pipelines
- The task involves git operations (merge, rebase, push)
- You're not sure of the full scope of the change
- The task involves deleting anything

**Can skip plan mode for:**
- Single-line bug fixes
- Adding a comment
- Renaming a variable in one file
- Running tests

---

## The Approval Conversation Pattern

Here's what expert Claude Code usage looks like for a non-trivial task:

```
You:    "I need to add rate limiting to all API routes."

Claude: [enters plan mode, explores codebase]
        "Here's my plan:
        1. Install `express-rate-limit`
        2. Create /middleware/rate-limit.ts with configurable limits
        3. Apply to all routes in /routes/index.ts
        4. Add tests in /tests/middleware/rate-limit.test.ts
        Should I proceed?"

You:    "Good, but: limits should be configurable per-route, not global.
        Also add a Redis store for distributed deployments.
        Check if we already have a Redis client somewhere."

Claude: [re-explores, revises plan]
        "Found a Redis client at /lib/redis.ts. Updated plan:
        1. Install `express-rate-limit` and `rate-limit-redis`
        2. Create /middleware/rate-limit.ts using existing Redis client
        3. Export a factory function `createRateLimit(options)` for per-route config
        4. Update routes to use per-route limits as documented in your API spec
        5. Tests covering both in-memory and Redis store modes
        Proceed?"

You:    "Yes, proceed."
```

This back-and-forth costs 2 minutes. Getting it wrong and undoing it costs 20.

---

## Permission Modes

Claude Code has different permission levels that affect what it does automatically:

- **Default mode** — Asks before taking risky actions (destructive ops, git pushes, etc.)
- **Auto-approve** — Can be configured for specific tools in settings
- **Read-only** — Claude can only read files, not write

For most work, keep the default mode. It's the right balance of autonomy and control.

---

## Practical Exercise

Take a non-trivial task in a real project and practice the full plan-mode workflow:

1. Ask Claude to plan the implementation (without doing it)
2. Read the plan carefully
3. Find at least one thing to push back on or refine
4. Get Claude to revise the plan
5. Approve and let it implement
6. Reflect: did the final result match your expectations better than if you'd just said "go"?
