# Lesson 07: Custom Skills (Slash Commands)

## What Are Skills?

Skills are reusable prompts that you can invoke with a `/command` shortcut. They're stored as markdown files and loaded into context when triggered.

Instead of typing the same long prompt repeatedly, you define it once as a skill and invoke it with `/skill-name`.

---

## Where Skills Live

Skills can be defined at two levels:

**Global skills** (available in every project):
```
~/.claude/commands/
  commit.md
  review-pr.md
  explain.md
  debug.md
```

**Project skills** (available only in that project):
```
<project-root>/.claude/commands/
  deploy.md
  seed-db.md
  generate-api-docs.md
```

---

## Anatomy of a Skill File

A skill is a markdown file. Its contents become the prompt that Claude receives when you invoke it.

**`~/.claude/commands/commit.md`:**
```markdown
Create a git commit for the staged changes.

Follow these rules exactly:
1. Run `git diff --staged` to see what's staged
2. Run `git log --oneline -5` to understand recent commit style
3. Write a conventional commit message: `type(scope): description`
   - Types: feat, fix, chore, docs, refactor, test, style
   - Keep the description under 72 characters
   - Add a body if the change needs explanation
4. Create the commit (never use --no-verify)
5. Show the commit hash when done

Do not push. Do not ask for confirmation — just commit.
```

Now you can just type `/commit` instead of explaining all this every time.

---

## Practical Skill Examples

### `/review` — Code Review

```markdown
Review the changes in the current branch against main.

Steps:
1. Run `git diff main...HEAD` to see all changes
2. For each changed file, evaluate:
   - Correctness: does the logic work?
   - Security: any OWASP top-10 issues?
   - Performance: any obvious bottlenecks?
   - Style: does it follow our CLAUDE.md conventions?
   - Test coverage: are the changes tested?
3. Output a structured review with sections: Summary, Issues (critical/major/minor), Suggestions
4. End with an overall verdict: Approve / Request Changes / Needs Discussion
```

### `/debug` — Systematic Debugging

```markdown
Help me debug the current issue using the scientific method.

1. Ask me to describe the symptom (what's happening vs. what should happen)
2. Identify what we know for certain vs. what we're assuming
3. Form 3 hypotheses ranked by likelihood
4. Design a test for the most likely hypothesis
5. After I run the test, evaluate results and either:
   a. Confirm the hypothesis and suggest a fix, or
   b. Eliminate it and move to the next hypothesis
6. Continue until root cause is found
```

### `/explain` — Deep Code Explanation

```markdown
Explain the following code to me as if I'm a smart developer who is unfamiliar with this specific codebase.

Structure your explanation as:
1. **What it does** (2-3 sentences, user-facing)
2. **How it works** (step-by-step walkthrough)
3. **Key design decisions** (why it's written this way)
4. **Gotchas** (anything surprising or non-obvious)
5. **Dependencies** (what other parts of the codebase it relies on)

Then ask me what I'd like to go deeper on.
```

### `/standup` — Daily Standup Prep

```markdown
Prepare my daily standup update.

1. Run `git log --since="yesterday" --author="$(git config user.name)" --oneline`
2. Check for any open PRs with `gh pr list --author @me`
3. Check for any PR review requests with `gh pr list --review-requested @me`
4. Summarize in standup format:
   - Yesterday: [what I completed]
   - Today: [what I'm working on based on open PRs/branches]
   - Blockers: [any PRs waiting for review]

Keep it concise — standup format, not an essay.
```

### `/perf` — Performance Analysis

```markdown
Analyze the performance characteristics of the code I'm about to show you.

Evaluate:
1. **Time complexity** — Big O for key operations
2. **Space complexity** — Memory usage patterns
3. **Database** — N+1 queries, missing indexes, unnecessary fetches
4. **Network** — Unnecessary round trips, missing caching
5. **Bottlenecks** — The single biggest performance problem

Then prioritize: what should be optimized first and why?
```

---

## Skills with Arguments

Skills can reference `$ARGUMENTS` to include whatever you typed after the slash command.

**`~/.claude/commands/issue.md`:**
```markdown
Create a GitHub issue for the following:

$ARGUMENTS

Requirements for the issue:
- Title: clear, actionable, starts with a verb
- Body: include context, steps to reproduce (if bug), acceptance criteria
- Labels: suggest appropriate labels based on type
- Use `gh issue create` to actually create it

Show me the issue URL when done.
```

Usage: `/issue Users can't log in on mobile Safari`

---

## Tips for Writing Great Skills

**Start with the outcome, not the steps.** What should Claude have produced when the skill completes?

**Include quality criteria.** "Write a good commit message" is vague. Spell out what good means.

**Specify the output format.** Should Claude give you a list? A table? Show code? Just make changes silently?

**Handle the common edge cases.** What should happen if there's nothing staged? If the tests fail?

**Iterate on your skills.** The first version won't be perfect. Use a skill a few times and refine it.

---

## Organizing Your Skills Library

As your skills grow, organize them:

```
~/.claude/commands/
  git/
    commit.md
    review.md
    standup.md
  dev/
    debug.md
    perf.md
    explain.md
  project/
    deploy.md
    changelog.md
```

Note: subdirectory skills are invoked as `/git/commit` or `/dev/debug`.

---

## Practical Exercise

Create 3 skills tailored to your actual workflow:

1. One for a repetitive git operation you do often
2. One for a type of code review or analysis you frequently need
3. One for something project-specific

Use each one 3 times and refine based on the results.
