# Lesson 09: Parallelization & Efficiency

## Why This Matters

Claude is capable of doing multiple things simultaneously using sub-agents and parallel tool calls. Expert users leverage this to dramatically speed up complex tasks.

---

## What Claude Can Parallelize Internally

When Claude makes multiple tool calls that don't depend on each other, it sends them in the same response. You see this when Claude does several file reads at once, or runs multiple searches simultaneously.

You can encourage this behavior:

> "Read these three files in parallel: `auth.ts`, `session.ts`, and `middleware/auth.ts`."

Or:

> "Simultaneously: (1) find all files that import from `utils/crypto`, (2) find all places where `Buffer.from` is called without encoding, (3) check if we have any tests for the crypto module."

---

## The Task Tool: True Parallelism

Claude can launch sub-agents to handle independent tasks truly in parallel. This is powerful for:

- Running tests while reading documentation
- Exploring multiple parts of a codebase simultaneously
- Doing research while planning implementation

### How to trigger it

Ask Claude to do multiple independent things at once:

> "I need you to do these two things in parallel:
> 1. Research how to implement WebSocket reconnection with exponential backoff
> 2. Map out all the files in our current socket handling code
>
> Then combine both when planning our implementation."

Claude will launch sub-agents for each task and report back when both are done.

---

## Batching Your Requests

Instead of one request per change, batch independent changes:

**Inefficient:**
```
You: "Add a phone field to the User type"
Claude: [does it]
You: "Add validation for phone numbers"
Claude: [does it]
You: "Add phone to the user registration form"
Claude: [does it]
```

**Efficient:**
```
You: "Add phone number support across the stack:
     1. Add `phone?: string` to the User type in types/user.ts
     2. Add E.164 format validation in lib/validators.ts
     3. Add the field to the registration form in components/RegisterForm.tsx
     4. Add it to the DB schema in prisma/schema.prisma
     Do all of these in one go."
```

Same result, one round trip.

---

## Background Tasks

For long-running tasks (running tests, building, linting), Claude can launch them in the background while continuing to work:

> "Run the test suite in the background while you implement the next feature. Let me know if tests fail when they complete."

---

## The Research-Then-Implement Pattern

For complex features, split into two phases:

**Phase 1: Pure research (fast, parallel)**
> "Before writing any code, research:
> 1. How our existing payment system is structured (read the services/payment directory)
> 2. What Stripe's webhook API looks like for subscription events
> 3. Whether we have any existing webhook handling infrastructure
>
> Give me a summary and your proposed approach."

**Phase 2: Focused implementation (informed)**
> "Based on your research, implement the Stripe webhook handler. You know where everything is."

Phase 1 is cheap (reads only). Phase 2 is fast (no exploration needed).

---

## Avoiding Bottlenecks

Some things must happen sequentially. Recognize them:

- Run tests → fix failures → run tests again (can't parallelize)
- Read file → understand it → edit it (read must come first)
- Create DB migration → run migration → update code (ordered)

But these can be parallelized:
- Read file A AND read file B (independent reads)
- Write test file AND write implementation file (independent writes)
- Run linter AND run type checker (independent validations)

When you structure your requests, think about what has dependencies vs. what's independent.

---

## Chunking for Speed

For large refactors, don't ask Claude to do everything at once. But also don't make it do one file at a time. Find the right chunk size:

**Too small:** "Update `user.service.ts` to use the new error types."
(Then: "Now do `billing.service.ts`." Then: "Now do `auth.service.ts`.")

**Too large:** "Update every service file in the project to use the new error types."
(Claude might lose track of what it's done, hit context limits, produce inconsistent results)

**Right size:** "Update all service files in `/src/services/` to use the new error types. There are 7 of them. Process them in batches of 3."

---

## Session Efficiency Tips

**Start with a summary request.** At the beginning of complex sessions:
> "Read the relevant files and give me a 3-sentence summary of the current state before we start making changes."

This primes Claude's context efficiently.

**Reuse established context.** Once Claude has read a file, don't ask it to read it again. Reference it by name:
> "In the `UserService` you read earlier, add..."

**Ask for summaries, not full files.** When you need to understand something:
> "Summarize the key data flow in `payment.service.ts` in bullet points — don't show me the whole file."

**Spot-check instead of full review.** After a large change:
> "Show me just the changed sections of each file, not the full file contents."

---

## Practical Exercise

Take a task that would normally require 5-6 back-and-forth exchanges and batch it into 1-2 requests.

Think through:
- What parts are independent? (do these together)
- What parts have dependencies? (do these in the right order)
- What research is needed before coding? (do this first, separately)

Compare the total time spent vs. your normal approach.
