# Lesson 11: Troubleshooting Common Issues

When things go wrong with Claude Code, here's how to diagnose and fix them.

---

## Issue 1: Claude Keeps Asking Questions It Should Know

**Symptom:** Every session, Claude asks "What's your tech stack?" or "Where should I put this file?"

**Diagnosis:** Missing or incomplete CLAUDE.md

**Fix:**
1. Read your CLAUDE.md: `cat CLAUDE.md` or ask Claude "Show me the CLAUDE.md"
2. Add the missing context:
   - Tech stack
   - Project structure
   - Key file locations
3. Test in a new session

**Prevention:** After any session where Claude asked repetitive questions, add the answers to CLAUDE.md.

---

## Issue 2: Claude Made the Wrong Architectural Decision

**Symptom:** Claude put business logic in a route handler, used the wrong pattern, or made a choice that violates your conventions.

**Diagnosis:** Your architecture rules aren't explicit enough in CLAUDE.md

**Fix:**
1. Correct the immediate problem
2. Add explicit architecture rules to CLAUDE.md:
   ```markdown
   ## Architecture Rules
   - Never put business logic in route handlers — use services
   - Never query the database from services — use repositories
   - All cross-service communication uses the event bus
   ```
3. Consider adding a DECISIONS.md file (see Lesson 10) to explain *why* these rules exist

**Prevention:** When you correct Claude's architectural decisions, immediately capture the rule in CLAUDE.md.

---

## Issue 3: Claude Went Down the Wrong Path

**Symptom:** Claude is implementing something complex but it's not what you wanted. You realize this 10 minutes in.

**Diagnosis:** Didn't use Plan Mode for a non-trivial task

**Fix:**
1. **Stop immediately.** Say: "Stop. Let's restart this approach."
2. Ask: "Before making any changes, show me your plan for what you were about to do."
3. Review the plan, identify where it diverged from your intent
4. Correct the plan: "Actually, I want you to [correct approach]"
5. Get approval before proceeding

**Prevention:** Use Plan Mode (`/plan` or "plan this first") for anything touching more than 2 files.

---

## Issue 4: Claude Can't Find Files or Code

**Symptom:** Claude says "I don't see any files that handle X" but you know they exist.

**Diagnosis:** Several possible causes:
- Files are in unexpected locations
- Code uses non-standard naming
- Claude searched with the wrong terms

**Fix:**
1. Tell Claude exactly where to look: "Check `/src/services/payment/stripe-webhook.ts`"
2. If using non-standard naming, add a glossary to CLAUDE.md:
   ```markdown
   ## Project Glossary
   - "Settlement" = our term for payout processing (not "payout")
   - "Merchant" = businesses using our platform (not "customer")
   ```
3. Add key file locations to CLAUDE.md:
   ```markdown
   ## Key Files
   - Payment webhooks: /src/services/payment/stripe-webhook.ts
   - Auth: /src/middleware/auth.ts
   ```

**Prevention:** Maintain a "Key Files" section in CLAUDE.md. Update it when you add important new files.

---

## Issue 5: Claude's Changes Broke Tests

**Symptom:** Claude made changes, now tests are failing.

**Diagnosis:** Normal! This is part of the workflow.

**Fix:**
1. Say: "The tests are failing. Read the test output and fix the failures."
2. Claude will read the errors and fix them
3. If tests fail again, repeat until green
4. If stuck in a loop (3+ rounds of fixing), say: "Explain why the tests keep failing. What's the root cause?"

**Prevention:**
- Use Test-Driven Development: write tests first, let Claude make them pass (Lesson 10)
- Add to CLAUDE.md: "Always run tests after making changes. Fix all failures before considering the task complete."

---

## Issue 6: Claude Made Too Many Changes

**Symptom:** You asked for a small fix, Claude refactored half the codebase.

**Diagnosis:** Didn't constrain the scope

**Fix:**
1. Review the changes: `git diff`
2. Revert: `git checkout .` (if not committed) or `git reset --hard HEAD~1` (if committed)
3. Re-request with explicit scope: "Fix only the null check on line 47. Don't refactor anything else."

**Prevention:**
- Be explicit about scope in your prompts
- Add to CLAUDE.md: "Never refactor code unless explicitly asked to refactor."
- Use Plan Mode to catch scope creep before implementation

---

## Issue 7: Claude Used the Wrong Tool/Library

**Symptom:** Claude installed a library you don't want, or used npm instead of bun.

**Diagnosis:** Your tool preferences aren't in CLAUDE.md

**Fix:**
1. Uninstall the wrong dependency
2. Tell Claude: "We use [correct tool]. Always use that."
3. Add to CLAUDE.md:
   ```markdown
   ## Development Tools
   - Package manager: bun (never npm or yarn)
   - Test runner: Vitest (never Jest)
   - HTTP client: fetch (never axios)
   ```

**Prevention:** Put all tool preferences in global `~/.claude/CLAUDE.md` so they apply to every project.

---

## Issue 8: Claude Forgot Context from Earlier in the Session

**Symptom:** Claude asks about something you discussed 10 messages ago.

**Diagnosis:** Long session with lots of back-and-forth. Context is getting lost.

**Fix:**
1. **Start a new session.** Context windows have limits.
2. Before starting fresh, ask: "Summarize what we accomplished this session. What should I remember?"
3. Save important context to MEMORY.md: "Remember that we use the repository pattern with Prisma."

**Prevention:**
- Break large tasks into focused sessions
- Use MEMORY.md to persist important discoveries
- When starting a new session, say: "Read your memory for this project and tell me what you know."

---

## Issue 9: Hook Failed / Permission Denied

**Symptom:** Claude tries to do something, you see an error about a hook failing.

**Diagnosis:** Your hooks (configured in `~/.claude/settings.json`) are blocking or erroring.

**Fix:**
1. Check which hook failed: look at the error message
2. Debug the hook:
   ```bash
   # Test the hook manually
   echo '{"tool_name":"Bash","tool_input":{"command":"test"}}' | ~/.claude/hooks/your-hook.sh
   ```
3. Fix the hook script or disable it temporarily:
   - Open `~/.claude/settings.json`
   - Comment out the problematic hook
4. Restart Claude Code

**Prevention:**
- Keep hooks simple and fast
- Always exit 0 unless intentionally blocking
- Test hooks before enabling them (see Lesson 05)

---

## Issue 10: Claude Won't Stop Talking / Too Verbose

**Symptom:** Claude writes essays when you want brief answers.

**Diagnosis:** No communication preferences set

**Fix:**
1. In the current conversation: "Be more concise. Give me bullet points, not paragraphs."
2. For all future sessions, add to `~/.claude/CLAUDE.md`:
   ```markdown
   ## Communication Style
   - Be concise. Prefer bullet points over paragraphs.
   - Show code, not descriptions of code.
   - Skip pleasantries. Get straight to the solution.
   - Never use emojis.
   ```

**Prevention:** Set communication preferences in global CLAUDE.md once, benefit forever.

---

## Issue 11: MCP Server Not Working

**Symptom:** You configured an MCP server but Claude can't use it.

**Diagnosis:** Several possible causes:
- Server not installed
- Missing environment variables
- Incorrect configuration

**Fix:**
1. Check the server is configured in `~/.claude/settings.json`
2. Test the server manually:
   ```bash
   # For npx-based servers
   npx -y @modelcontextprotocol/server-github
   ```
3. Check environment variables are set correctly
4. Restart Claude Code after config changes

**Prevention:**
- Test MCP servers independently before adding to settings
- Use absolute paths in server commands
- Keep credentials in environment variables, not settings.json (see Lesson 06)

---

## Issue 12: Git Operations Failed

**Symptom:** Claude tried to commit/push and got an error.

**Common causes:**

**Nothing to commit:**
- Fix: Stage files first, or tell Claude "stage the changes and commit"

**Pre-commit hook failed:**
- Fix: Claude should fix the issues (linting, tests) and commit again
- Never use `--no-verify` unless explicitly intended

**Merge conflict:**
- Fix: Ask Claude to "resolve the merge conflict in [file]"
- Or resolve manually and then continue

**Permission denied on push:**
- Fix: Check git remote authentication
- Ensure you have push access to the branch

**Prevention:**
Add to CLAUDE.md:
```markdown
## Git Workflow
- Always run tests before committing
- Never use --no-verify
- Never force push to main
- Create feature branches for all work
```

---

## General Debugging Approach

When something goes wrong:

**1. Stop and diagnose** - Don't let Claude keep trying the same thing
   > "Stop. Explain what went wrong and why."

**2. Get Claude's hypothesis** - Let it reason through the problem
   > "Before fixing, what do you think caused this error?"

**3. Fix the immediate issue** - Solve the problem at hand

**4. Prevent recurrence** - Update CLAUDE.md or MEMORY.md
   > "What should we add to CLAUDE.md to prevent this next time?"

---

## When to Start Fresh

Sometimes the best fix is a clean slate. Start a new session when:
- You've been going back and forth for 15+ messages without progress
- Claude is confused about the current state
- You've pivoted the task significantly
- The conversation has too many failed attempts cluttering context

Before starting fresh:
1. Ask Claude: "Summarize what we learned and what still needs to be done"
2. Save important context to MEMORY.md
3. Start the new session with clear context from the summary

---

## Getting Help

If you're truly stuck:

1. **Check Claude Code docs:** `/help` in the CLI
2. **Search GitHub issues:** https://github.com/anthropics/claude-code/issues
3. **File a bug report:** Include the error message and steps to reproduce
4. **Check your memory:** `cat ~/.claude/projects/*/memory/MEMORY.md` — sometimes the answer is already saved

---

## The Troubleshooting Mindset

Every issue is a learning opportunity. When something goes wrong:
- ✅ Fix it
- ✅ Understand why it happened
- ✅ Add a rule to prevent it
- ✅ Move on

Your CLAUDE.md file is a living record of every mistake you won't make twice.
